﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using Microsoft.Win32;
using System.Diagnostics;

namespace InternalsVisibleToInjector
{
    public partial class Form1 : Form
    {
        public Form1()
        {
            InitializeComponent();
            DetectPaths();
        }

        private void DetectPaths()
        {
            ilasmPath.Text = Path.Combine(System.Runtime.InteropServices.RuntimeEnvironment.GetRuntimeDirectory(), "ilasm.exe");
            RegistryKey key = Registry.LocalMachine.OpenSubKey(@"SOFTWARE\Microsoft\.NETFramework");
            if (key != null)
            {
                string value = (string)key.GetValue("sdkInstallRootv2.0");
                if (!string.IsNullOrEmpty(value))
                {
                    ildasmPath.Text = value + @"bin\ildasm.exe";
                }
            }
        }

        private void selectIldasmPath_Click(object sender, EventArgs e)
        {
            openFileDialog.Filter = "ILDASM.EXE|ILDASM.EXE";
            if (openFileDialog.ShowDialog() == DialogResult.OK)
            {
                ildasmPath.Text = openFileDialog.FileName;
            }
        }

        private void selectIlasmPath_Click(object sender, EventArgs e)
        {
            openFileDialog.Filter = "ILASM.EXE|ILASM.EXE";
            if (openFileDialog.ShowDialog() == DialogResult.OK)
            {
                ilasmPath.Text = openFileDialog.FileName;
            }
        }

        private void selectAssembly_Click(object sender, EventArgs e)
        {
            openFileDialog.Filter = "Assemblies (*.exe;*.dll)|*.dll;*.exe";
            if (openFileDialog.ShowDialog() == DialogResult.OK)
            {
                assemblyPath.Text = openFileDialog.FileName;
                outputAssemblyPath.Text = string.Format("{0}-Injected{1}", 
                    Path.Combine(Path.GetDirectoryName(assemblyPath.Text), 
                    Path.GetFileNameWithoutExtension(assemblyPath.Text)), 
                    Path.GetExtension(assemblyPath.Text));
            }
        }

        private void selectOutputAssemblyPath_Click(object sender, EventArgs e)
        {
            saveFileDialog.Filter = "Assemblies (*.exe;*.dll)|*.dll;*.exe";
            if (saveFileDialog.ShowDialog() == DialogResult.OK)
            {
                outputAssemblyPath.Text = saveFileDialog.FileName;
            }
        }

        private void processButton_Click(object sender, EventArgs e)
        {
            if (!File.Exists(ildasmPath.Text))
            {
                MessageBox.Show("ILDASM.EXE not found.", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }

            if (!File.Exists(ilasmPath.Text))
            {
                MessageBox.Show("ILASM.EXE not found.", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }

            if (!File.Exists(assemblyPath.Text))
            {
                MessageBox.Show(string.Format("Assembly {0} not found.", assemblyPath.Text), "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }

            if (string.IsNullOrEmpty(assemblyToInject.Text.Trim()))
            {
                MessageBox.Show("Assembly name to inject is blank.", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }

            if (string.IsNullOrEmpty(outputAssemblyPath.Text.Trim()))
            {
                MessageBox.Show("Output assembly name is blank.", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }

            if (File.Exists(outputAssemblyPath.Text.Trim()))
            {
                DialogResult result = MessageBox.Show("Output assembly already exists and will be overwritten.  Are you sure you want to continue?", "Output Assembly Exists", MessageBoxButtons.YesNo, MessageBoxIcon.Question);
                if (result == DialogResult.Yes)
                {
                    File.Delete(outputAssemblyPath.Text.Trim());
                }
                else
                {
                    return;
                }
            }

            // Create temp folder
            string tempPath = Path.Combine(Path.GetTempPath(), Guid.NewGuid().ToString());
            Directory.CreateDirectory(tempPath);

            try
            {
                this.Cursor = Cursors.WaitCursor;

                string baseName = Path.GetFileNameWithoutExtension(assemblyPath.Text);
                string fileName = Path.GetFileName(assemblyPath.Text);
                string ilPath = Path.Combine(tempPath, fileName + ".il");
                string resourcePath = Path.Combine(tempPath, fileName + ".res");

                // Disassemble
                string arguments = string.Format("/OUT=\"{0}\" /TEXT /NOBAR /RAWEH /QUOTEALLNAMES /UTF8 /FORWARD \"{1}\"", ilPath, assemblyPath.Text);
                ProcessStartInfo psi = new ProcessStartInfo(ildasmPath.Text, arguments);
                psi.CreateNoWindow = true;
                psi.UseShellExecute = false;
                psi.ErrorDialog = true;
                psi.WorkingDirectory = tempPath;
                Process process = new Process();
                process.StartInfo = psi;
                process.Start();
                process.WaitForExit();

                // Validate IL file existence
                if (!File.Exists(ilPath))
                {
                    throw new FileNotFoundException("Disassebled IL file not found.");
                }

                // Create IL to insert
                StringBuilder bldr = new StringBuilder();
                bldr.Append("  .custom instance void ['mscorlib']'System.Runtime.CompilerServices'.'InternalsVisibleToAttribute'::.ctor(string) = ( 01 00 ");
                byte[] chars = System.Text.Encoding.ASCII.GetBytes(assemblyToInject.Text.Trim());
                byte length = (byte)chars.Length;
                bldr.Append(length.ToString("X2"));
                bldr.Append(' ');
                foreach (byte bt in chars)
                {
                    bldr.Append(bt.ToString("X2"));
                    bldr.Append(' ');
                }
                bldr.Append("00 00 )");
                string toinsert = bldr.ToString();

                // Read file into memory
                List<string> lines = new List<string>(File.ReadAllLines(ilPath, Encoding.UTF8));

                // Find assembly segment and modify
                string lookfor = string.Format(".assembly '{0}'", baseName);
                for (int i = 0; i < lines.Count; i++)
                {
                    if ((string.Compare(lines[i], lookfor, true) == 0) && lines[i+1].StartsWith("{"))
                    {
                        int j = i + 1;
                        while (!(lines[j]).StartsWith("}"))
                        {
                            j++;
                        }
                        j--;
                        while (lines[j].StartsWith("  .ver") || lines[j].StartsWith("  .hash algorithm"))
                        {
                            j--;
                        }
                        lines.Insert(j + 1, toinsert);
                    }
                }

                // Write output
                File.WriteAllLines(ilPath, lines.ToArray(), Encoding.UTF8);

                // Reassemble
                string dllString = string.Empty;
                if (Path.GetExtension(assemblyPath.Text).ToLower() == ".dll")
                {
                    dllString = "/dll ";
                }
                string resourceString = string.Empty;
                if (File.Exists(resourcePath))
                {
                    resourceString = string.Format("/resource=\"{0}\"", resourcePath);
                }
                arguments = string.Format("/OUTPUT=\"{0}\" /nologo /quiet {3} {1} \"{2}\"", outputAssemblyPath.Text.Trim(), resourceString, ilPath, dllString);

                psi = new ProcessStartInfo(ilasmPath.Text, arguments);
                psi.CreateNoWindow = true;
                psi.UseShellExecute = false;
                psi.ErrorDialog = true;
                psi.WorkingDirectory = tempPath;
                process = new Process();
                process.StartInfo = psi;
                process.Start();
                process.WaitForExit();

                // If output assembly exists, assume success
                if (File.Exists(outputAssemblyPath.Text))
                {
                    MessageBox.Show("Operation appears to have completed successfully!", "Success", MessageBoxButtons.OK, MessageBoxIcon.Information);
                }
                else
                {
                    MessageBox.Show("Reassembly failed for an unknown reason.", "Failure", MessageBoxButtons.OK, MessageBoxIcon.Asterisk);
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
            finally
            {
                Directory.Delete(tempPath, true);
                this.Cursor = Cursors.Default;
            }
        }
    }
}
